//==============================================================================================================
//
//  Transmit (egress) Speed Test: Measure how fast bytes egress out the network interface (Wi-Fi or Ethernet)
//
//  This code is the intellectual property of Jerry Jongerius (duckware.com).  Before using, just ask permission.
//
//  See https://www.duckware.com/support/contact.html
//
//--------------------------------------------------------------------------------------------------------------
//
//  Transmission over Wi-Fi is very 'bursty'.  Nothing is transmitted for a while, and then a bunch of data can
//  be written over a very short period of time.  We presume this is just how the Wi-Fi driver interacts with the
//  SO_SNDBUF socket buffer (accepts data in 'bursts').  We call each 'pause+burst' a 'write-run'.  So all bytes
//  transmitted in the burst must actually be accounted for over the entire time frame of the 'write-run'.
//
//  BEWARE using this program on a 'metered' Internet connection
//
//--------------------------------------------------------------------------------------------------------------
//
//  TODO:
//    - a filter should be added 'at the end' to catch outlier speeds
//
//==============================================================================================================

import java.io.*;
import java.net.*;
import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.awt.*;

public class txrate extends Panel {

    //--------------------------------------------------------------------------------
    private static long g_tBoot = System.nanoTime();
    private static long tick() {
        return System.nanoTime()-g_tBoot;
        }
    //--------------------------------------------------------------------------------
    // milliseconds to ticks
    private static long ms2t( int ms ) {
        return ms*1000000L;
        }
    //--------------------------------------------------------------------------------
    // deci (1/10) ms
    private static double dms( long ns ) {
        return (ns/100000/10.0);
        }
    //--------------------------------------------------------------------------------
    private static String tos( int[] an ) {
        String ret = "";
        for (int loop=0; loop<an.length; ++loop) {
            ret += (loop>0?" ":"")+an[loop];
            }
        return ret;
        }
    //--------------------------------------------------------------------------------
    private static int imax( int[] an ) {
        int ret = 0;
        for (int loop=0; loop<an.length; ++loop) {
            if (an[loop]>=an[ret]) {
                ret = loop;
                }
            }
        return ret;
        }

    //--------------------------------------------------------------------------------
    private static long amax( long[] an ) {
        long ret = 0;
        for (int loop=0; loop<an.length; ++loop) {
            ret = Math.max(ret,an[loop]);
            }
        return ret;
        }
    //--------------------------------------------------------------------------------
    private static double amax( double[] av ) {
        double ret = 0;
        for (int loop=0; loop<av.length; ++loop) {
            ret = Math.max(ret,av[loop]);
            }
        return ret;
        }
    //--------------------------------------------------------------------------------
    private static long asum( long[] an ) {
        long ret = 0;
        for (int loop=0; loop<an.length; ++loop) {
            ret += an[loop];
            }
        return ret;
        }
    //----------------------------------------------------------------------
    private static String getProperty( String key ) {
        return System.getProperty(key);
        }
    //----------------------------------------------------------------------
    private static double d1( double d ) {
        return Math.floor(d*10)/10.0;
        }
    //----------------------------------------------------------------------
    private static double per( double d ) {
        return d1(100*d);
        }

    //================================================================================
    //================================================================================
    //================================================================================

    private AppFrame m_app;

    private int g_nDebugLevel=0;                  // debug level (0=off)
    private static final int MSRUN=20;            // max milliseconds allowed in a 'write run'
    private static final int MSSLOT=50;           // should equally divide into msDuration
    private DatagramChannel m_s;                  // UDP socket
    private ByteBuffer m_buffer;                  // packet to send
    private long m_tRunStart=0;                   // time that last run started
    private long m_tRunEnd=0;                     // time that last run ended
    private long m_tLastWrite=0;                  // time of last write that failed/worked (for debug print)
    private long m_tLastWriteWorked=0;            // time of last write that worked
    private long m_lMaxStall=0;                   // maximum time between writes that worked
    private int m_nNumStalls=0;                   // number of times 'write run' timed out
    private long m_tPushing=0;                    // time spend pushing bytes, but nothing was written

    private int m_nScaleInclude=10;

    //--------------------------------------------------------------------------------
    public static void main( String args[] ) {
        try {
            new txrate().go(args);
            }
        catch (IOException e) {
            e.printStackTrace();
            }
        System.exit(0);
        }

    //--------------------------------------------------------------------------------
    private void go( String args[] ) throws IOException {
        String host = null;
        int nPort = 21065;                 // default port
        int nPacketSize = 1400;            // default: 1400 bytes
        int msDuration = 2000;             // default: 2 seconds
        int nSoSndBuf = 256*1024;          // default: 256 KB
        boolean bLiveview = false;         // default: off

        // parse command line arguments
        for (int loop=0; loop<args.length; ++loop) {
            String arg = args[loop];
            if (arg.startsWith("-l:")) {
                nPacketSize = Integer.parseInt(arg.substring(3));
                }
            else if (arg.startsWith("-ms:")) {
                msDuration = Integer.parseInt(arg.substring(4));
                }
            else if (arg.startsWith("-port:")) {
                nPort = Integer.parseInt(arg.substring(6));
                }
            else if (arg.startsWith("-debug:")) {
                g_nDebugLevel = Integer.parseInt(arg.substring(7));
                }
            else if (arg.startsWith("-sndbuf:")) {
                nSoSndBuf = Integer.parseInt(arg.substring(8));
                }
            else if (arg.startsWith("-scale:")) {
                m_nScaleInclude = Integer.parseInt(arg.substring(7));
                }
            else if (arg.equals("-live")) {
                bLiveview = true;
                }
            else if (arg.startsWith("-")) {
                System.out.println( "??? "+arg );
                }
            else if (null==host) {
                host = arg;
                }
            }
        if (null==host) {
            System.out.println( "Duckware (R) txrate (TM) Version 4.1a" );
            System.out.println( "Copyright (c) 2006-2023 Duckware. All rights reserved.");
            System.out.println( "Java "+getProperty("java.vendor")+" "+getProperty("java.version")+" ["+getProperty("java.home")+"]" );
            System.out.println( "" );
            System.out.println( "txrate [-l:size] [-port:#] [-ms:#] [-debug:#] [-sndbuf:#] [-live] HOST" );
            System.out.println( "" );
            System.out.println( "    HOST: ip address of the local gateway" );
            System.out.println( "" );
            System.out.println( "  Options:" );
            System.out.println( "    -l:size    size of packets to send (default: 1400)" );
            System.out.println( "    -port:#    send packets to this UDP port (default: 21065)" );
            System.out.println( "    -ms:#      duration of test in milliseconds (default: 2000)" );
            System.out.println( "    -debug:#   set debug level 2|1 (default: 0, or off)" );
            System.out.println( "    -sndbuf:#  socket SO_SNDBUF value (default: 262144)" );
            System.out.println( "    -scale:#   include this mbps value in scale calculations" );
            System.out.println( "    -live      displays a real-time chart" );
            }
        else {
            m_app = new AppFrame(this);
            if (bLiveview) {
                m_app.show();
                }
            run( nSoSndBuf, host, nPort, nPacketSize, msDuration );
            while (m_app.isShowing()) {
                G.sleep(200);
                }
            }
        }

    //--------------------------------------------------------------------------------
    // Write a single packet to the UDP socket
    private int doWritePacket( boolean bRunning ) throws IOException {
        m_buffer.rewind();
        int nSent = m_s.write(m_buffer);
        long tNow = tick();
        long tDelta = tNow-m_tLastWrite;
        if (g_nDebugLevel>=2) {
            System.out.println(tNow+"\t"+tDelta+"\t"+nSent);
            }
        m_tLastWrite = tNow;

        // Calculate 'max stall' metric (but only if running / past the 'startup' time)
        if (nSent>0) {
            m_lMaxStall = Math.max(m_lMaxStall,bRunning?tNow-m_tLastWriteWorked:0);
            m_tLastWriteWorked = tNow;
            }
        else {
            m_tPushing += tDelta;
            }

        return nSent;
        }

    //--------------------------------------------------------------------------------
    // Measures bytes sent over a 'write run', where a 'write run' is (but with caveat that run has a max time):
    //   (a) not being able to send any packets
    //   (b) followed by sending a burst of sending packets (up until next 'not being able to send')
    private int doWriteRun( boolean bRunning ) throws IOException {
        int nTotalSent=0;

        m_tRunStart = m_tRunEnd;    // the start of this run picks up where last run left off
        long tSending=0;            // time of first send
        long t1 = tick();
        long tEnd = m_tLastWrite;
        boolean bSleep=false;
        // limit run length to (a) measure stalls, and (b) prevent outright hangs
        // allows up to MSRUN of writing - helps filter pass to find outlier situations
        while ((tick()-Math.max(t1,tSending))<ms2t(MSRUN)) {
            tEnd = m_tLastWrite;
            int nSent = doWritePacket(bRunning);
            if (tSending>0 && 0==nSent) {
                bSleep = true;
                break;
                }
            nTotalSent += nSent;
            if (0==tSending && nSent>0) {
                tSending = m_tLastWrite;
                }
            }
        m_tRunEnd = tEnd;
        long delta=m_tRunEnd-m_tRunStart;
 
        //**************************
        if (bSleep) {
            // EXPERIMENT sleeping for a very short period of time to reduce
            // CPU overhead.  But this is ONLY for 'slow' connections.  For
            // fast connections, sleeping for 1ms is 'too long'.  As compensation
            // could consider significantly increasing SO_SNDBUF
            //G.sleep(1);
            }
        //**************************

        // debug
        if (g_nDebugLevel>=1) {
            System.out.println("*** ["+m_tRunStart+"-"+m_tRunEnd+"] "+nTotalSent+" in "+(delta/1000000.0) );
            }

        // BUGBUG: Sometimes a Wi-Fi driver just eats packets and does not send them out 'over the air'.
        // Why?  That results in impossibly high throughput spikes.  We attempt to filter out those
        // spikes.
        if (bRunning && calcmbps(nTotalSent, delta)>5000 && delta>ms2t(2)) {
            System.out.println( "ERR: "+nTotalSent+" in "+(delta/1000000.0)+" is too fast - filtered out" );
            nTotalSent = 0;
            }

        // a 'write run' that ends with NO bytes sent is considered a 'stalled' write-run
        if (0==nTotalSent && bRunning) {
            ++m_nNumStalls;
            }

        return nTotalSent;
        }

    //--------------------------------------------------------------------------------
    // convert raw values into shape -- each value changed to "percentage of max value"
    private int[] makeShape( long[] alSent ) {
        int anRet[] = new int[alSent.length];
        long nMax = amax(alSent);
        if (nMax>0) {
            for (int loop=0; loop<alSent.length; ++loop) {
                anRet[loop] = (int)(100.0*alSent[loop]/nMax);
                }
            }
        return anRet;
        }
    //--------------------------------------------------------------------------------
    // 'contention' metric is a percentage: area above the graph divided by bounding rectangle around graph
    private double calcContention( long alSent[] ) {
        long lBox = Math.max(1,amax(alSent)*alSent.length);
        return (double)(lBox-asum(alSent))/lBox;
        }
    //--------------------------------------------------------------------------------
    private double calcStalled( int msDuration ) {
        return (double)m_nNumStalls/((msDuration+MSRUN-1)/MSRUN);
        }
    //--------------------------------------------------------------------------------
    private double calcmbps( long lBytes, long ticks ) {
        return ticks>0 ? 8.0*lBytes/1000000/(ticks/1000000000.0) : 0;
        }

    //--------------------------------------------------------------------------------
    public byte[] createPacket( int nSize ) {
        byte[] ret = new byte[nSize];
        String s = "** network interface Tx speed test **";
        int len = s.length();
        for (int loop=0; loop<ret.length; loop+=len) {
            s.getBytes( 0, Math.min(len,ret.length-loop), ret, loop );
            }
        return ret;
        }
    //--------------------------------------------------------------------------------
    private int getSendBufferSize() throws IOException {
        return m_s.getOption(StandardSocketOptions.SO_SNDBUF);
        }
    //--------------------------------------------------------------------------------
    // setup UDP socket in non-blocking mode
    private void doSetupSocket( int nSoSndBuf, InetSocketAddress addr, int nPacketSize ) throws IOException {
        m_s = DatagramChannel.open();
        m_s.configureBlocking(false);
        if (g_nDebugLevel>=1) {
            System.out.println( "default SO_SNDBUF: "+ getSendBufferSize() );
            }
        m_s.setOption(StandardSocketOptions.SO_SNDBUF, nSoSndBuf);
        m_s.bind(null);
        m_s.connect(addr);
        m_buffer = ByteBuffer.wrap(createPacket(nPacketSize));
        }
    //--------------------------------------------------------------------------------
    // STARTUP:
    //   - don't collect metrics on startup
    //   - this should also fill ALL egress buffers (especially SO_SNDBUF)
    //   - synchronize to a 'write run' boundary
    private long doStartup( int nSoSndBuf ) throws IOException {
        long tNow = tick();
        m_tLastWrite = tNow;   // 'prime' internal metrics
        m_tRunEnd = tNow;      // 'prime' internal metrics

        long t1 = tNow;
        int nWritten=0;
        for (int loop=0; loop<3 || (tick()-t1)<ms2t(100) || nWritten<nSoSndBuf; ++loop) {
            nWritten += doWriteRun(false);
            }
        long tStartup = tick()-t1;

        if (g_nDebugLevel>=1) {
            System.out.println("startup in "+dms(tStartup));
            }

        m_tPushing = 0;
        return tStartup;
        }

    //--------------------------------------------------------------------------------
    private void run( int nSoSndBuf, String host, int nPort, int nPacketSize, int msDuration ) throws IOException {
        System.out.println( "Running..." );
        Thread me = Thread.currentThread();
        me.setPriority(Thread.MAX_PRIORITY);

        long lBytesSent=0;
        long alSent[] = new long[(int)(msDuration/MSSLOT)];

        // startup
        doSetupSocket( nSoSndBuf, new InetSocketAddress(host, nPort), nPacketSize );
        long tStartup = doStartup(nSoSndBuf);
        m_tGraphStart = m_tRunEnd;

        // MEASURE network egress transmit rate
        long t1 = m_tRunEnd;
        long ta = t1;
        for (int loop=0; (ta-t1)<ms2t(msDuration); ++loop) {
            int nSent = doWriteRun(true);
            if (nSent>0) {
                addSent( alSent, nSent, ta-t1, m_tRunEnd-t1, MSSLOT );
                lBytesSent += nSent;
                }
            setHaveLiveBytes( nSent, alSent, (int)((m_tRunEnd-t1)/ms2t(MSSLOT)) );
            ta = m_tRunEnd;
            }
        long tDelta = ta-t1;
        double mbps = calcmbps(lBytesSent,tDelta);
        setHaveResult( mbps );
        doRepaint(true);
        me.setPriority(Thread.NORM_PRIORITY);

        // output results
        String SEP="=";
        System.out.println( "ipTarget"    +SEP+ host );
        System.out.println( "nPort"       +SEP+ nPort );
        System.out.println( "SO_SNDBUF"   +SEP+ getSendBufferSize() );
        System.out.println( "nPacketSize" +SEP+ nPacketSize );
        System.out.println( "msStartup"   +SEP+ dms(tStartup) );
        System.out.println( "msDuration"  +SEP+ dms(tDelta) );
        System.out.println( "msPushing"   +SEP+ dms(m_tPushing) );
        System.out.println( "nBytesSent"  +SEP+ lBytesSent );
        System.out.println( "txshape"     +SEP+ tos(makeShape(alSent)) );
        System.out.println( "msMaxStall"  +SEP+ dms(m_lMaxStall) );
        System.out.println( "pStalled"    +SEP+ per(calcStalled(msDuration)) );
        System.out.println( "pContention" +SEP+ per(calcContention(alSent)) );
        System.out.println( "txrate"      +SEP+ d1(mbps) );

        m_s.close();  // Note: any packets remaining in SO_SNDBUF *are* still sent
        }

    //--------------------------------------------------------------------------------
    // pro-rate 'bytes sent' during a 'write run' over the entire 'write run' time frame
    private void addSent( long[] alSent, int nSent, long ta, long tb, int MSSLOT ) {
        if (tb>ta) {
            long tSlot = ms2t(MSSLOT);        // slot time/size in ticks
            int nSlot1 = (int)(ta/tSlot);     // write run starts in this slot
            int nSlot2 = (int)(tb/tSlot);     // ..and ends in this slot
            for (int loop=nSlot1; loop<=nSlot2; ++loop) {
                if (loop>=0 && loop<alSent.length) {
                    long t1 = Math.max((loop+0)*tSlot,ta);
                    long t2 = Math.min((loop+1)*tSlot,tb);
                    if (t1<t2) {
                        alSent[loop] += (long)((double)(t2-t1)/(tb-ta)*nSent);
                        }
                    }
                }
            }
        }

    //================================================================================
    //================================================================================
    //================================================================================

    // This chart code is a hack, just to get something displayed QUICKLY

    private long m_tGraphStart=0;      // time base for start of graph
    private long m_onbucket;           // used to update live mbps
    private int m_accum;
    private long m_alSent[];
    private int m_nAt;

    //----------------------------------------------------------------------
    private long m_nPaintBucket=0;
    private void doRepaint( boolean bNow ) {
        long on = tick()/ms2t(16);
        if (bNow || on!=m_nPaintBucket) {
            m_nPaintBucket = on;
            repaint();
            }
        }

    //----------------------------------------------------------------------
    int SS[] = new int[] {10,20,30,40,50,75,100,150,200,250,300,400,500,600,750,1000,2000};  // scale steps (must be x10 / gridlines)
    private int mapScale( double scale ) {
        for (int loop=0; loop<SS.length; ++loop) {
            if (scale<SS[loop]) {
                return SS[loop];
                }
            }
        return (int)scale;
        }
    //----------------------------------------------------------------------
    private void setHaveLiveBytes( int nSent, long[] alSent, int nAt ) {
        m_alSent = alSent;
        m_nAt = nAt;

        long tBucket=ms2t(500);
        long on = (tick()-m_tGraphStart)/tBucket;
        if (on!=m_onbucket) {
            m_onbucket = on;
            setHaveResult( calcmbps(m_accum, tBucket) );
            m_accum = 0;
            }
        m_accum += nSent;
        doRepaint(false);
        }
    //----------------------------------------------------------------------
    private void setHaveResult( double mbps ) {
        setInfo( "TxRate: "+d1(mbps)+" Mbps" );
        doRepaint(false);
        }

    private String m_info="";
    //----------------------------------------------------------------------
    private void setInfo( String s ) {
        m_info = s;
        }

    //----------------------------------------------------------------------
    // OVERRIDE
    public void update( Graphics g ) {
        paint( g );
        }
    //----------------------------------------------------------------------
    // OVERRIDE
    public void paint( Graphics g ) {
        if (DoubleBuffer.go(this,g)) {
            paint2( g );
            }
        }
    //----------------------------------------------------------------------
    private void paint2( Graphics g ) {
        int xx = 10;
        int yy = 10;
        Dimension d = size();

        // gray background
        g.setColor( new Color(0xF0F0F0) );
        g.fillRect( 0, 0, d.width, d.height );

        // info string
        if (true) {
            g.setFont( new Font("Arial", Font.BOLD, 20) );
            FontMetrics fm = g.getFontMetrics();
            yy += fm.getAscent();
            g.setColor( Color.BLUE );
            g.drawString(""+m_info, xx, yy );
            yy += 8;
            }

        // chart
        if (null!=m_alSent) {
            g.setFont( new Font("Arial", Font.PLAIN, 12) );

            double ambps[] = new double[Math.min(m_nAt,m_alSent.length)];
            for (int loop=0; loop<ambps.length; ++loop) {
                ambps[loop] = d1(calcmbps(m_alSent[loop],ms2t(MSSLOT)));
                }
            int scale = mapScale(Math.max(m_nScaleInclude,amax(ambps)));

            int EX=10;                      // extra border pixels
            int gw = d.width-2*EX;          // graph width pixels
            int gh = d.height-yy-2*EX-20;   // graph height pixels
            int nDiv = m_alSent.length-1;   // horizontal divisions

            // inside of graph is white
            g.setColor( Color.WHITE );
            g.fillRect( EX, d.height-EX-gh, gw, gh );

            // horizontal gridlines / Y-axis labels
            for (int loop=0; loop<10; ++loop) {
                int y = d.height-EX- loop*gh/10;
                g.setColor( new Color(0xE0E0E0) );
                g.drawLine( EX, y, EX+gw, y );
                g.setColor( Color.BLACK );
                g.drawString( ""+(scale*loop/10), EX+4, y-2 );
                }

            // vertical line at live location
            if (m_nAt>0 && m_nAt<m_alSent.length) {
                g.setColor( Color.BLUE );
                int x = EX+gw*(m_nAt-1)/nDiv;
                g.drawLine( x, d.height-EX, x, d.height-EX-gh );
                }

            // polyline
            g.setColor( Color.BLUE );
            int xv[] = new int[ambps.length];
            int yv[] = new int[ambps.length];
            if (scale>0) {
                g.setFont( new Font("Arial", Font.PLAIN, 12) );
                g.setColor( Color.BLACK );
                g.drawString( "Mbps", EX, d.height-EX-gh-2 );
                for (int loop=0; loop<xv.length; ++loop) {
                    xv[loop] = EX+gw*loop/nDiv;
                    yv[loop] = d.height-EX- (int)(gh*ambps[loop]/scale);
                    }
                }
            g.drawPolyline( xv, yv, xv.length );

            // black border around graph
            g.setColor( Color.BLACK );
            g.drawRect( EX, d.height-EX-gh, gw, gh );
            }
        }
    }
